@once
<style>
    .chat-popup-container {
        position: relative; /* Changed from fixed to relative */
        z-index: 1000;
        display: flex;
        flex-direction: column;
        align-items: flex-end; /* Align popup under the button */
    }
    .chat-popup-toggle {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        color: white;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        box-shadow: none; /* Remove shadow for navbar integration */
    }
    .chat-popup {
        position: absolute;
        top: 100%; /* Position below the button */
        right: 0;
        width: 320px;
        height: 400px;
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        display: none;
        flex-direction: column;
        overflow: hidden;
        margin-top: 10px; /* Space between button and popup */
    }
    .chat-popup.active {
        display: flex;
    }
    .chat-popup-header {
        padding: 10px 15px;
        background-color: #f8f9fa;
        border-bottom: 1px solid #e9ecef;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .chat-popup-body {
        flex: 1;
        overflow-y: auto;
        padding: 10px;
    }
    .chat-popup-body.contacts {
        padding: 0;
    }
    .chat-popup-footer {
        padding: 10px;
        border-top: 1px solid #e9ecef;
        display: flex;
    }
    .chat-popup-input {
        flex: 1;
        border: 1px solid #ddd;
        border-radius: 20px;
        padding: 8px 15px;
        margin-right: 10px;
    }
    .chat-popup-send {
        background-color: #2180f3;
        color: white;
        border: none;
        border-radius: 50%;
        width: 36px;
        height: 36px;
        cursor: pointer;
    }
    .contact-item {
        padding: 10px 15px;
        border-bottom: 1px solid #e9ecef;
        cursor: pointer;
        display: flex;
        align-items: center;
    }
    .contact-item:hover {
        background-color: #f8f9fa;
    }
    .contact-avatar {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        margin-right: 10px;
    }
    .contact-info {
        flex: 1;
    }
    .contact-name {
        font-weight: bold;
    }
    .contact-last-message {
        font-size: 0.8rem;
        color: #6c757d;
        text-overflow: ellipsis;
        white-space: nowrap;
        overflow: hidden;
        max-width: 200px;
    }
    .message-item {
        margin-bottom: 10px;
        display: flex;
        flex-direction: column;
    }
    .message-item.from-me {
        align-items: flex-end;
    }
    .message-item.from-other {
        align-items: flex-start;
    }
    .message-bubble {
        padding: 8px 12px;
        border-radius: 18px;
        max-width: 80%;
    }
    .message-item.from-me .message-bubble {
        background-color: #2180f3;
        color: white;
    }
    .message-item.from-other .message-bubble {
        background-color: #f1f3f4;
    }
    .message-time {
        font-size: 0.7rem;
        color: #6c757d;
        margin-top: 2px;
    }
    .back-button {
        background: none;
        border: none;
        cursor: pointer;
        color: #6c757d;
        margin-right: 10px;
    }

    /* Styles for separate chat windows */
    #chatWindowsContainer {
        position: fixed;
        bottom: 20px;
        left: 20px;
        right: 90px; /* Space for the main chat button */
        display: flex;
        flex-direction: row;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 10px;
        pointer-events: none; /* Allow clicks to pass through container */
        z-index: 999;
    }

    .chat-window {
        width: 300px;
        height: 350px;
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        display: flex;
        flex-direction: column;
        overflow: hidden;
        pointer-events: auto; /* Enable clicks on the window */
        margin-bottom: 10px;
        position: relative; /* For absolute positioning of elements inside */
    }

    .chat-window.minimized {
        overflow: hidden;
    }
    .chat-window-header {
        padding: 8px 12px;
        background-color: #f8f9fa;
        border-bottom: 1px solid #e9ecef;
        display: flex;
        justify-content: space-between;
        align-items: center;
        cursor: pointer;
    }

    .chat-window-title {
        font-weight: bold;
        display: flex;
        align-items: center;
        text-overflow: ellipsis;
        white-space: nowrap;
        overflow: hidden;
        max-width: 220px;
    }

    .chat-window-title img {
        width: 24px;
        height: 24px;
        border-radius: 50%;
        margin-right: 8px;
    }

    .chat-window-actions {
        display: flex;
        gap: 5px;
    }

    .chat-window-action {
        background: none;
        border: none;
        cursor: pointer;
        color: #6c757d;
        padding: 2px 5px;
        font-size: 14px;
    }

    .chat-window-body {
        flex: 1;
        overflow-y: auto;
        padding: 10px;
        background-color: #fff;
    }

    .chat-window-footer {
        padding: 8px;
        border-top: 1px solid #e9ecef;
        display: flex;
        background-color: #fff;
    }

    .chat-window-input {
        flex: 1;
        border: 1px solid #ddd;
        border-radius: 20px;
        padding: 6px 12px;
        margin-right: 8px;
        font-size: 0.9rem;
    }

    .chat-window-attachment {
        margin-top: 8px;
        margin-right: 5px;
        cursor: pointer;
    }

    .chat-window-send {
        background-color: #2180f3;
        color: white;
        border: none;
        border-radius: 50%;
        width: 32px;
        height: 32px;
        cursor: pointer;
        font-size: 0.8rem;
    }

    .loading-messages {
        padding: 10px;
        color: #6c757d;
        text-align: center;
    }

    .message-count {
        position: absolute;
        top: -5px;
        right: -5px;
        background-color: #ff4136;
        color: white;
        border-radius: 50%;
        width: 20px;
        height: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 12px;
    }
</style>
<link href="{{ asset('css/chatify/style.css') }}" rel="stylesheet">
<div class="chat-popup-container">
    <div class="chat-popup" id="chatPopup">
        <div class="chat-popup-header">
            <div id="chatHeader">
                <h4>Contacts</h4>
            </div>
            <button id="closePopup">✕</button>
        </div>
        <div class="chat-popup-body contacts" id="contactsContainer">
            <!-- Contacts will be loaded here -->
            <div id="loadingContacts">Loading contacts...</div>
        </div>
        <div class="chat-popup-body messages" id="messagesContainer" style="display: none;">
            <!-- Messages will be loaded here -->
        </div>
        <div class="chat-popup-footer" id="chatFooter" style="display: none;">
            <label for="attachment" class="attachment-btn" style="margin-top: 12px; margin-right: 5px">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21.44 11.05l-9.19 9.19a6 6 0 0 1-8.49-8.49l9.19-9.19a4 4 0 0 1 5.66 5.66l-9.2 9.19a2 2 0 0 1-2.83-2.83l8.49-8.48"></path>
                </svg>
            </label>
            <input type="file" id="attachment" style="display: none;">
            <input type="text" class="chat-popup-input" id="messageInput" placeholder="Type a message...">
            <button class="chat-popup-send" id="sendButton">➤</button>
        </div>
    </div>

    <!-- Container for separate chat windows -->
    <div id="chatWindowsContainer"></div>

    <div class="chat-popup-toggle" id="chatToggle">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
        </svg>
    </div>
</div>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        let activeChats = {};
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const maxWindows = 4; // Maximum number of windows visible at once

        // Create bubble container
        const bubbleContainer = document.createElement('div');
        bubbleContainer.id = 'chat-bubbles-container';
        document.body.appendChild(bubbleContainer);

        // Add bubble container styles
        const styleElement = document.createElement('style');
        styleElement.textContent = `
        #chat-bubbles-container {
            position: fixed;
            bottom: 90px;
            right: 20px;
            display: flex;
            flex-direction: column;
            gap: 10px;
            z-index: 999;
        }

        .chat-bubble {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background-color: #fff;
            box-shadow: 0 2px 10px rgba(0,0,0,0.2);
            cursor: pointer;
            position: relative;
            overflow: hidden;
            transition: transform 0.3s ease;
        }

        .chat-bubble:hover {
            transform: scale(1.1);
        }

        .chat-bubble img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .chat-bubble .notification-dot {
            position: absolute;
            top: 0;
            right: 0;
            width: 15px;
            height: 15px;
            background-color: #ff4136;
            border-radius: 50%;
            border: 2px solid #fff;
        }
    `;
        document.head.appendChild(styleElement);

        // Toggle main chat popup
        document.getElementById('chatToggle').addEventListener('click', function() {
            document.getElementById('chatPopup').classList.toggle('active');
            if (document.getElementById('chatPopup').classList.contains('active')) {
                loadContacts();
            }
        });

        // Close main chat popup
        document.getElementById('closePopup').addEventListener('click', function() {
            document.getElementById('chatPopup').classList.remove('active');
        });

        // Load contacts
        function loadContacts() {
            fetch('{{route("chatify.contact")}}')
                .then(response => response.json())
                .then(data => {
                    let contactsHTML = '';
                    data.contacts.forEach(function(contact) {
                        contactsHTML += `
                        <div class="contact-item" data-user-id="${contact.user.id}" data-user-name="${contact.user.name}" data-user-avatar="${contact.avatar}">
                            <img src="${contact.avatar}" class="contact-avatar" alt="${contact.user.name}">
                            <div class="contact-info">
                                <div class="contact-name">${contact.user.name}</div>
                                <div class="contact-last-message">${contact.lastMessage || 'No messages yet'}</div>
                            </div>
                        </div>
                    `;
                    });
                    document.getElementById('contactsContainer').innerHTML = contactsHTML;

                    // Add click event for contacts
                    document.querySelectorAll('.contact-item').forEach(item => {
                        item.addEventListener('click', function() {
                            const userId = this.getAttribute('data-user-id');
                            const userName = this.getAttribute('data-user-name');
                            const userAvatar = this.getAttribute('data-user-avatar');

                            // Create a new chat window
                            createChatWindow(userId, userName, userAvatar);

                            // Optionally hide the contacts popup after creating a window
                            document.getElementById('chatPopup').classList.remove('active');
                        });
                    });
                })
                .catch(error => {
                    document.getElementById('contactsContainer').innerHTML = '<div class="p-3">Error loading contacts</div>';
                });
        }

        // Create a new chat window
        function createChatWindow(userId, userName, userAvatar) {
            // If chat window already exists, just focus on it and return
            if (activeChats[userId]) {
                const existingWindow = document.getElementById(activeChats[userId]);
                if (existingWindow) {
                    // If minimized, restore it
                    if (existingWindow.classList.contains('minimized')) {
                        toggleMinimizeChatWindow(userId);
                    }

                    // Bring it to front
                    bringWindowToFront(userId);
                    return;
                }
            }

            // Create a unique ID for this chat window
            const chatWindowId = `chat-window-${userId}`;
            const attachmentId = `attachment-${userId}`;

            // Create chat window HTML
            const chatWindowHTML = `
            <div class="chat-window" id="${chatWindowId}" data-user-id="${userId}">
                <div class="chat-window-header">
                    <div class="chat-window-title">
                        <img src="${userAvatar}" alt="${userName}">
                        <span>${userName}</span>
                    </div>
                    <div class="chat-window-actions">
                        <button class="chat-window-action minimize-btn" title="Minimize">—</button>
                        <button class="chat-window-action close-btn" title="Close">✕</button>
                    </div>
                </div>
                <div class="chat-window-body" id="messages-${userId}">
                    <div class="loading-messages">Loading messages...</div>
                </div>
                <div class="chat-window-footer">
                    <label for="${attachmentId}" class="chat-window-attachment">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21.44 11.05l-9.19 9.19a6 6 0 0 1-8.49-8.49l9.19-9.19a4 4 0 0 1 5.66 5.66l-9.2 9.19a2 2 0 0 1-2.83-2.83l8.49-8.48"></path>
                        </svg>
                    </label>
                    <input type="file" id="${attachmentId}" style="display: none;">
                    <input type="text" class="chat-window-input" id="input-${userId}" placeholder="Type a message...">
                    <button class="chat-window-send" id="send-${userId}">➤</button>
                </div>
            </div>
        `;

            // Add to the container
            document.getElementById('chatWindowsContainer').insertAdjacentHTML('beforeend', chatWindowHTML);

            // Add to active chats
            activeChats[userId] = chatWindowId;

            // Store original dimensions for restoration
            const windowElement = document.getElementById(chatWindowId);
            windowElement.dataset.originalHeight = '350px';

            // Add event listeners
            document.querySelector(`#${chatWindowId} .close-btn`).addEventListener('click', function(e) {
                e.stopPropagation();
                closeChatWindow(userId);
            });

            document.querySelector(`#${chatWindowId} .minimize-btn`).addEventListener('click', function(e) {
                e.stopPropagation();
                toggleMinimizeChatWindow(userId);
            });

            document.querySelector(`#${chatWindowId} .chat-window-header`).addEventListener('click', function(e) {
                if (!e.target.closest('.chat-window-action')) {
                    if (windowElement.classList.contains('minimized')) {
                        toggleMinimizeChatWindow(userId);
                    } else {
                        bringWindowToFront(userId);
                    }
                }
            });

            document.getElementById(`send-${userId}`).addEventListener('click', function() {
                sendMessageFromWindow(userId);
            });

            document.getElementById(`input-${userId}`).addEventListener('keypress', function(e) {
                if (e.which === 13) {
                    sendMessageFromWindow(userId);
                }
            });

            document.getElementById(attachmentId).addEventListener('change', function() {
                const messageInput = document.getElementById(`input-${userId}`);
                if (this.files[0]) {
                    messageInput.placeholder = `File selected: ${this.files[0].name}`;
                } else {
                    messageInput.placeholder = 'Type a message...';
                }
            });

            // Making the window draggable
            makeWindowDraggable(chatWindowId);

            // Load messages for this user
            loadMessagesForWindow(userId);

            // Bring this window to front
            bringWindowToFront(userId);

            // Auto-focus the input field
            setTimeout(() => {
                document.getElementById(`input-${userId}`).focus();
            }, 100);

            // Arrange windows if there are too many
            arrangeWindows();
        }

        // Make a window draggable
        function makeWindowDraggable(windowId) {
            const windowElement = document.getElementById(windowId);
            const header = windowElement.querySelector('.chat-window-header');

            let isDragging = false;
            let offsetX, offsetY;

            header.addEventListener('mousedown', function(e) {
                if (!e.target.closest('.chat-window-action')) {
                    isDragging = true;
                    offsetX = e.clientX - windowElement.getBoundingClientRect().left;
                    offsetY = e.clientY - windowElement.getBoundingClientRect().top;
                    windowElement.style.position = 'absolute';
                    bringWindowToFront(windowElement.getAttribute('data-user-id'));
                }
            });

            document.addEventListener('mousemove', function(e) {
                if (isDragging) {
                    const x = e.clientX - offsetX;
                    const y = e.clientY - offsetY;

                    // Keep window within viewport bounds
                    const maxX = window.innerWidth - windowElement.offsetWidth;
                    const maxY = window.innerHeight - windowElement.offsetHeight;

                    windowElement.style.left = Math.max(0, Math.min(maxX, x)) + 'px';
                    windowElement.style.top = Math.max(0, Math.min(maxY, y)) + 'px';
                }
            });

            document.addEventListener('mouseup', function() {
                isDragging = false;
            });
        }

        // Close a chat window
        function closeChatWindow(userId) {
            if (activeChats[userId]) {
                const windowElement = document.getElementById(activeChats[userId]);
                // Remove bubble if exists
                removeChatBubble(userId);

                if (windowElement) {
                    // Add closing animation if desired
                    windowElement.style.animation = 'fadeOut 0.2s';
                    setTimeout(() => {
                        windowElement.remove();
                        delete activeChats[userId];
                        arrangeWindows();
                    }, 200);
                }

                // Clear any interval for this chat
                if (activeChats.refreshIntervals && activeChats.refreshIntervals[userId]) {
                    clearInterval(activeChats.refreshIntervals[userId]);
                    delete activeChats.refreshIntervals[userId];
                }
            }
        }

        // Toggle minimize/maximize a chat window
        function toggleMinimizeChatWindow(userId) {
            if (activeChats[userId]) {
                const windowElement = document.getElementById(activeChats[userId]);
                if (windowElement) {
                    // Toggle minimized class
                    windowElement.classList.toggle('minimized');
                    const isMinimized = windowElement.classList.contains('minimized');

                    if (isMinimized) {
                        // Before hiding, store original height and position if not already set
                        if (!windowElement.dataset.originalHeight) {
                            windowElement.dataset.originalHeight = windowElement.offsetHeight + 'px';
                        }
                        if (windowElement.style.left) {
                            windowElement.dataset.originalLeft = windowElement.style.left;
                        }
                        if (windowElement.style.top) {
                            windowElement.dataset.originalTop = windowElement.style.top;
                        }

                        // Hide window
                        windowElement.style.display = 'none';

                        // Create bubble for minimized chat
                        createChatBubble(userId,
                            windowElement.querySelector('.chat-window-title img').src,
                            windowElement.querySelector('.chat-window-title span').textContent);

                        // Clear any notifications on the window
                        clearNotification(userId);
                    } else {
                        // Show window
                        windowElement.style.display = 'flex';

                        // Remove bubble
                        removeChatBubble(userId);

                        // Restore window body and footer
                        const body = windowElement.querySelector('.chat-window-body');
                        const footer = windowElement.querySelector('.chat-window-footer');

                        body.style.display = 'block';
                        footer.style.display = 'flex';

                        // Restore original dimensions and position
                        if (windowElement.dataset.originalHeight) {
                            windowElement.style.height = windowElement.dataset.originalHeight;
                        }
                        if (windowElement.dataset.originalLeft) {
                            windowElement.style.left = windowElement.dataset.originalLeft;
                        }
                        if (windowElement.dataset.originalTop) {
                            windowElement.style.top = windowElement.dataset.originalTop;
                        }

                        // Scroll to bottom when maximizing
                        setTimeout(() => {
                            body.scrollTop = body.scrollHeight;
                        }, 10);

                        // Bring window to front
                        bringWindowToFront(userId);
                    }
                }
            }
        }

        // Create chat bubble for minimized window
        function createChatBubble(userId, userAvatar, userName) {
            // Check if bubble already exists
            if (document.getElementById(`chat-bubble-${userId}`)) {
                return;
            }

            const bubble = document.createElement('div');
            bubble.id = `chat-bubble-${userId}`;
            bubble.className = 'chat-bubble';
            bubble.setAttribute('data-user-id', userId);
            bubble.setAttribute('title', userName);

            const avatar = document.createElement('img');
            avatar.src = userAvatar;
            avatar.alt = userName;

            bubble.appendChild(avatar);

            // Add click event to restore window
            bubble.addEventListener('click', function() {
                toggleMinimizeChatWindow(userId);
            });

            // Add to bubble container
            bubbleContainer.appendChild(bubble);
        }

        // Remove chat bubble
        function removeChatBubble(userId) {
            const bubble = document.getElementById(`chat-bubble-${userId}`);
            if (bubble) {
                bubble.remove();
            }
        }

        // Show notification on bubble
        function showBubbleNotification(userId) {
            const bubble = document.getElementById(`chat-bubble-${userId}`);
            if (!bubble) return;

            // Check if notification already exists
            if (!bubble.querySelector('.notification-dot')) {
                const notificationDot = document.createElement('div');
                notificationDot.className = 'notification-dot';
                bubble.appendChild(notificationDot);
            }
        }

        // Bring window to front (by manipulating z-index)
        function bringWindowToFront(userId) {
            const windows = document.querySelectorAll('.chat-window');
            windows.forEach(window => {
                window.style.zIndex = '1000';
            });

            if (activeChats[userId]) {
                const windowElement = document.getElementById(activeChats[userId]);
                if (windowElement) {
                    windowElement.style.zIndex = '1001';
                }
            }
        }

        // Arrange windows to prevent overflow
        function arrangeWindows() {
            const windowsList = Object.values(activeChats)
                .map(id => document.getElementById(id))
                .filter(el => el && !el.classList.contains('minimized'));

            const containerWidth = window.innerWidth - 100; // Accounting for margins

            // If there are more windows than our max visible limit, arrange them
            if (windowsList.length > maxWindows) {
                const spacing = 10;
                const windowWidth = 300 + spacing;

                // How many windows can fit in a row
                const windowsPerRow = Math.floor(containerWidth / windowWidth);

                windowsList.forEach((window, index) => {
                    // Determine row and column
                    const row = Math.floor(index / windowsPerRow);
                    const col = index % windowsPerRow;

                    // Position window
                    window.style.position = 'absolute';
                    window.style.left = (col * windowWidth) + 'px';

                    // Position from bottom
                    const bottomPosition = 20 + (row * 50); // For minimized stacking
                    window.style.bottom = bottomPosition + 'px';

                    // Minimize windows beyond the first few
                    if (index >= maxWindows) {
                        const userId = window.getAttribute('data-user-id');
                        if (!window.classList.contains('minimized')) {
                            toggleMinimizeChatWindow(userId);
                        }
                    }
                });
            }
        }

        // Load messages for a specific window
        function loadMessagesForWindow(userId) {
            const messagesContainer = document.getElementById(`messages-${userId}`);

            fetch(`{{url('chatify-popup/messages')}}?user_id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    messagesContainer.innerHTML = data.messages;
                    // Scroll to bottom
                    messagesContainer.scrollTop = messagesContainer.scrollHeight;

                    // Set up periodic refresh for new messages
                    setupMessageRefresh(userId);
                })
                .catch(error => {
                    messagesContainer.innerHTML = '<div class="p-3">Error loading messages</div>';
                });
        }

        // Setup refresh interval for new messages
        function setupMessageRefresh(userId) {
            // Check for new messages every 10 seconds
            const refreshId = setInterval(() => {
                // Only refresh if window still exists and is not closed
                if (activeChats[userId] && document.getElementById(activeChats[userId])) {
                    refreshMessages(userId);
                } else {
                    clearInterval(refreshId);
                }
            }, 10000);

            // Store interval ID to clear when window is closed
            if (!activeChats.refreshIntervals) activeChats.refreshIntervals = {};
            activeChats.refreshIntervals[userId] = refreshId;
        }

        // Refresh messages for a specific window
        function refreshMessages(userId) {
            const windowElement = document.getElementById(activeChats[userId]);
            if (!windowElement) return;

            const messagesContainer = document.getElementById(`messages-${userId}`);
            const wasAtBottom = messagesContainer.scrollHeight - messagesContainer.scrollTop <= messagesContainer.clientHeight + 50;

            fetch(`{{url('chatify-popup/messages')}}?user_id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.messages !== messagesContainer.innerHTML) {
                        messagesContainer.innerHTML = data.messages;

                        // If was at bottom before, scroll back to bottom
                        if (wasAtBottom) {
                            messagesContainer.scrollTop = messagesContainer.scrollHeight;
                        }

                        // If window is minimized, show notification on bubble
                        if (windowElement.classList.contains('minimized')) {
                            showBubbleNotification(userId);
                        }
                    }
                })
                .catch(error => {
                    // Silent error handling for background refresh
                    console.error("Error refreshing messages:", error);
                });
        }

        // Show notification dot for new message on minimized window
        function showMessageNotification(userId) {
            const windowElement = document.getElementById(activeChats[userId]);
            if (!windowElement) return;

            // If minimized to bubble, show notification there
            if (windowElement.classList.contains('minimized')) {
                showBubbleNotification(userId);
                return;
            }

            // Check if notification already exists
            let notif = windowElement.querySelector('.message-count');
            if (!notif) {
                // Create notification dot
                notif = document.createElement('div');
                notif.className = 'message-count';
                notif.textContent = '1';
                windowElement.appendChild(notif);
            } else {
                // Increment count
                const count = parseInt(notif.textContent) || 0;
                notif.textContent = count + 1;
            }
        }

        // Clear notification when window is maximized
        function clearNotification(userId) {
            const windowElement = document.getElementById(activeChats[userId]);
            if (windowElement) {
                const notif = windowElement.querySelector('.message-count');
                if (notif) {
                    notif.remove();
                }
            }

            // Also clear bubble notification
            const bubble = document.getElementById(`chat-bubble-${userId}`);
            if (bubble) {
                const bubbleNotif = bubble.querySelector('.notification-dot');
                if (bubbleNotif) {
                    bubbleNotif.remove();
                }
            }
        }

        // Send message from a chat window
        function sendMessageFromWindow(userId) {
            const messageInput = document.getElementById(`input-${userId}`);
            const message = messageInput.value.trim();
            const fileInput = document.getElementById(`attachment-${userId}`);

            if (message === '' && !fileInput.files[0]) return;

            const formData = new FormData();
            formData.append('_token', csrfToken);
            formData.append('user_id', userId);
            formData.append('message', message);

            // Append file if exists
            if (fileInput.files[0]) {
                formData.append('file', fileInput.files[0]);
            }

            // Optimistic UI update - add message immediately
            const messagesContainer = document.getElementById(`messages-${userId}`);
            const currentTime = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});

            if (message) {
                const tempMessageHTML = `
                <div class="message-item from-me">
                    <div class="message-bubble">${message}</div>
                    <div class="message-time">${currentTime}</div>
                </div>
            `;
                messagesContainer.insertAdjacentHTML('beforeend', tempMessageHTML);
                messagesContainer.scrollTop = messagesContainer.scrollHeight;
            }

            // Reset input fields
            messageInput.value = '';
            fileInput.value = '';
            messageInput.placeholder = 'Type a message...';

            // Send message to server
            fetch('{{route("chatify.send")}}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': csrfToken
                },
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    // Reload messages to ensure correct display
                    loadMessagesForWindow(userId);
                })
                .catch(error => {
                    console.error("Error sending message:", error);
                    // Handle error, maybe add error message to UI
                });
        }

        // Keep original chat popup functionality
        let currentUserId = null;

        // Send message from main popup
        document.getElementById('sendButton').addEventListener('click', function() {
            sendMessage();
        });

        // Send message on Enter key from main popup
        document.getElementById('messageInput').addEventListener('keypress', function(e) {
            if (e.which === 13) {
                sendMessage();
            }
        });

        // Send message function for main popup
        function sendMessage() {
            const messageInput = document.getElementById('messageInput');
            const message = messageInput.value.trim();
            const fileInput = document.getElementById('attachment');

            if ((message === '' && !fileInput.files[0]) || currentUserId === null) return;

            const formData = new FormData();
            formData.append('_token', csrfToken);
            formData.append('user_id', currentUserId);
            formData.append('message', message);

            // Append file if exists
            if (fileInput.files[0]) {
                formData.append('file', fileInput.files[0]);
            }

            fetch('{{route("chatify.send")}}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': csrfToken
                },
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    messageInput.value = '';
                    fileInput.value = '';
                    // Reload messages to show the new message
                    loadMessages(currentUserId);
                });
        }

        // Add file input change handler to show selected file
        document.getElementById('attachment').addEventListener('change', function() {
            const messageInput = document.getElementById('messageInput');
            if (this.files[0]) {
                messageInput.placeholder = `File selected: ${this.files[0].name}`;
            } else {
                messageInput.placeholder = 'Type a message...';
            }
        });

        // Handle window resize to rearrange chat windows
        window.addEventListener('resize', function() {
            arrangeWindows();
        });
    });
</script>
@endonce
